<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Article;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ArticleController extends Controller
{
    public function index()
    {
        $articles = Article::with('category','author')->latest()->paginate(20);
        return view('admin.articles.index', compact('articles'));
    }

    public function create()
    {
        $categories = Category::orderBy('name')->get();
        $article = new Article();
        return view('admin.articles.create', compact('categories','article'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title'        => 'required|string|max:255',
            'slug'         => 'nullable|string|max:255|unique:articles,slug',
            'excerpt'      => 'nullable|string|max:500',
            'body'         => 'required|string',
            'status'       => 'required|in:draft,published',
            'category_id'  => 'nullable|exists:categories,id',
            'meta_title'   => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:300',
        ]);

        $data['slug'] = $data['slug'] ?: Str::slug($data['title']).'-'.time();
        $data['author_id'] = auth()->id();
        $data['published_at'] = $data['status'] === 'published' ? now() : null;

        Article::create($data);

        return redirect()->route('articles.index')->with('success','تم إنشاء المقال بنجاح');
    }

    public function edit(Article $article)
    {
        $categories = Category::orderBy('name')->get();
        return view('admin.articles.edit', compact('article','categories'));
    }

    public function update(Request $request, Article $article)
    {
        $data = $request->validate([
            'title'        => 'required|string|max:255',
            'slug'         => 'nullable|string|max:255|unique:articles,slug,'.$article->id,
            'excerpt'      => 'nullable|string|max:500',
            'body'         => 'required|string',
            'status'       => 'required|in:draft,published',
            'category_id'  => 'nullable|exists:categories,id',
            'meta_title'   => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:300',
        ]);

        $data['slug'] = $data['slug'] ?: $article->slug;
        $data['published_at'] = $data['status'] === 'published'
            ? ($article->published_at ?? now())
            : null;

        $article->update($data);

        return redirect()->route('articles.index')->with('success','تم تحديث المقال بنجاح');
    }

    public function destroy(Article $article)
    {
        $article->delete();
        return redirect()->route('articles.index')->with('success','تم حذف المقال بنجاح');
    }
}
